/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx_ibcast.c,v 1.13 2005/06/29 00:23:16 eugene Exp $";

#include <pthread.h>
#include <netinet/in.h>
#include <unistd.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "mx_auto_config.h"
#include "myriexpress.h"
#include "internal.h"
#include "packet.h"

static inline void 
hex_dump(void *buf, uint32_t len) {
  int i;
  for (i=0; i<len; i++) {
    printf("%02x ", ((unsigned char *)buf)[i]);
    if (i%16==15) printf("\n");
  }
}


mx_return_t
mx_ibcast (mx_endpoint_t endpoint,
           mx_segment_t *segments_list,
           uint32_t segments_count,
           mx_endpoint_addr_t *addr_list,
           uint32_t addr_count,
           uint32_t root_index,
	   uint64_t match_info,
           void *context,
           mx_request_t *request)
{
  return mx_ibcast_with_type(endpoint, segments_list, segments_count,
                             addr_list, addr_count,
                             root_index, match_info,
                             context, request, MX_SR_TYPE_BCAST);
}

struct mx_post *
mx_create_bcast_post(struct mx_endpoint *pp,
		     mx_segment_t *segments_list,
		     uint32_t segments_count,
		     mx_endpoint_addr_t *addresss_list,
		     uint32_t addresss_count,
		     uint32_t root_index,
		     uint64_t match_info,
		     void* context,
		     mx_sr_type_t type)
{
  struct mx_post *post;
  struct mx_lib_bcast *bcast;

  post = mx_new_post(pp, MX_POST_TYPE_BCAST, context);
  if (post == NULL) {
    return NULL;
  }
  
  bcast = &post->ts.bcast;
  bcast->length = mx_segment_list_len(segments_list, segments_count);
  
  bcast->seg_list = mx_new_segment_list(segments_list, segments_count);
  if (bcast->seg_list == NULL) {
    mx_destroy_post(post);
    return NULL;
  }
  
  bcast->seg_cnt = segments_count;
  bcast->dests = addresss_list;
  bcast->dest_count = addresss_count;
  bcast->dest_skip = root_index;
  bcast->match_info = match_info;
  bcast->type = type;
  
  return post;
}

mx_return_t
mx_ibcast_with_type (mx_endpoint_t endpoint,
                     mx_segment_t *segments_list,
                     uint32_t segments_count,
                     mx_endpoint_addr_t *addresss_list,
                     uint32_t addresss_count,
                     uint32_t root_index,
		     uint64_t match_info,
                     void *context,
                     mx_request_t *request,
                     mx_sr_type_t type)
{
  struct mx_endpoint *pp = endpoint;
  mx_return_t status = MX_SUCCESS;
  uint32_t i;
  struct mx_post *post;

  
  if (root_index >= addresss_count) {
    status = MX_BAD_ROOT;
    goto abort_with_lock;
  }

  printf("my_address\n");
  hex_dump(pp->my_address.stuff, 16);
  for (i = 0; i < addresss_count; ++i) {
    printf("dest[%d]\n", i);
    hex_dump(addresss_list[i].stuff,16);
    if (memcmp (pp->my_address.stuff, addresss_list[i].stuff,
		sizeof(pp->my_address.stuff)) == 0) {
      break;
    }
  }
  if (i == addresss_count) {
    status = MX_BAD_BAD_BAD;
    goto abort_with_lock;
  }

  if (i == root_index) {
    /* I'm the sender. */
    pthread_mutex_lock(&Mx_tx_lock);
  
    post = mx_create_bcast_post(pp, segments_list, segments_count,
				addresss_list, addresss_count,
				root_index, match_info,
				context, type);
    if (post == NULL) {
      status = MX_NO_RESOURCES;
      pthread_mutex_unlock(&Mx_tx_lock);
      goto abort_with_lock;
    }
    MX_LIST_INSERT(&pp->bcast_list, post);
    MX_DEBUG_PRINT(MX_DEBUG_TCP_LIB,("posting bcast(0x%08x%08x)\n", MX_U32(match_info), MX_L32(match_info)));

    *request = (mx_request_t) post;
    pthread_cond_signal(&Mx_send_cond);
    pthread_mutex_unlock(&Mx_tx_lock);

  }
  else {
    /* I'm one of the receivers. */
    MX_DEBUG_PRINT(MX_DEBUG_TCP_LIB,("posting receive(bcast)\n"));
    mx_irecv_with_type(endpoint,
                       segments_list,
                       segments_count,
		       match_info,
                       MX_MATCH_MASK_NONE,
                       context,
                       request, type);
  }

 abort_with_lock:
  return status;
}

uint32_t
mx_try_bcast(struct mx_post *post)
{
  int s;
  int i;
  int n;
  int rc;
  struct mx_lib_bcast *bcast;
  struct mx_send_hdr snd_hdr;
  uint64_t x;
  int j;
  mx_pkt_type_t pkt_type;

  /* get ptr to send struct */
  bcast = &post->ts.bcast;

  for (j = 0; j < bcast->dest_count; ++j) {
    if (j != bcast->dest_skip) {
      x = *(uint64_t*)&bcast->dests[j];
      MX_DEBUG_PRINT(MX_DEBUG_TCP_LIB,("trying send to 0x%08x%08x\n", MX_U32(x), MX_L32(x)));
      /* See if we can connect to this address */
      s = get_sock_for_address(post->endpoint, bcast->dests[j]);
      
      /* if no connection (yet), return unsuccessful */
      if (s == -1) {
	return 0;
      }
      
      /* got connected, so perform the send */
      
      /* first a header */
      if (post->type != MX_POST_TYPE_BCAST) {
	/* XXX */
	MX_DEBUG_PRINT(MX_DEBUG_TCP_LIB,("this is not a bcast!!!\n"));
	exit(1);
      }
      switch (bcast->type) {
      case MX_SR_TYPE_BCAST:
	pkt_type = MX_PKT_TYPE_BCAST;
	break;
      case MX_SR_TYPE_BARRIER_ACK:
	pkt_type = MX_PKT_TYPE_BARRIER_ACK;
	break;
      default:
	MX_DEBUG_PRINT(MX_DEBUG_TCP_LIB,("unknown bcast post type %d\n", bcast->type));
	exit(1);
	break;
      }
      
      mx_send_header(s, pkt_type, post->endpoint,
		     bcast->length + sizeof (snd_hdr));
      
      /* then a send sub-header */
      snd_hdr.match_a = htonl(bcast->match_info >> 32);
      snd_hdr.match_b = htonl(bcast->match_info & 0xffffffff);
      rc = mx_sock_write(s, &snd_hdr, sizeof(snd_hdr));
      
      /* then the body of the message */
      for (i=0; i<bcast->seg_cnt; ++i) {
	n = mx_sock_write(s,
                          bcast->seg_list[i].segment_ptr,
                          bcast->seg_list[i].segment_length);
	if (n < 0){
          perror("write error\n");
          /* XXX this could be better */
          exit(1);
        }
      }
    }
  }
  MX_DEBUG_PRINT(MX_DEBUG_TCP_LIB,("completed bcast, len = %d, handle = 0x%p\n", bcast->length,
	    post));

  /* send successful! */
  return 1;
}
